##
# Procedural model of a Diamond HK36 TTC electrical system.
# Reference: Flight Manual, Maintenance Manual and Engine Operation Manual


##
# Initialize internal values
#

var vbus_volts = 0.0;
var ebus1_volts = 0.0;
var ebus2_volts = 0.0;

var ammeter_ave = 0.0;

##
# Initialize Switches and Circuit breakers
#
var switches		=	props.globals.getNode("/controls/switches", 1);
var master_switch	=	switches.getNode("master",	1);
var mode_switch		=	switches.getNode("mode",	1);	# 0 = SOARING; 1 = POWERED FLIGHT
var acl_switch		=	switches.getNode("acl",		1);
var pos_switch		=	switches.getNode("pos",		1);
var land_switch		=	switches.getNode("ldg",		1);
var instr_switch	=	switches.getNode("instr",	1);
var fp_switch		=	switches.getNode("fuelpump",	1);
var starter		=	switches.getNode("starter",	1);

var breakers		=	props.globals.getNode("/controls/circuit-breakers", 1);
var cb = {
		main	:	breakers.getNode("main",	1),
		comm0	:	breakers.getNode("comm[0]",	1),
		gps	:	breakers.getNode("gps",		1),
		xpdr	:	breakers.getNode("transponder",	1),
		att	:	breakers.getNode("att-gyro",	1),
		engine1	:	breakers.getNode("engine[0]",	1),
		engine2	:	breakers.getNode("engine[1]",	1),
	};
	
##
# Initialize properties used to determine electrical load
#
var com_ptt = props.globals.getNode("/instrumentation/comm[0]/ptt", 1);
var com_start = props.globals.getNode("/instrumentation/comm[0]/start", 1);
var vario_vol = props.globals.getNode("/instrumentation/ilec-sc7/volume", 1);
var vario_aud = props.globals.getNode("/instrumentation/ilec-sc7/audio", 1);
var vario_read = props.globals.getNode("/instrumentation/ilec-sc7/te-reading-mps", 1);
var flarm_receive = props.globals.getNode("/instrumentation/FLARM/receive-internal", 1);
var xpdr_on	=	props.globals.getNode("/instrumentation/it-gtx327/internal/system-alive", 1);
var att_spin	=	props.globals.getNode("/instrumentation/attitude-indicator/spin", 1);

##
# Battery model class.
#

#	AMM 2.6.1
#		voltage: 12V
#		capacity: 18Ah

var BatteryClass = {};

BatteryClass.new = func {
	var obj = { parents : [BatteryClass],
		ideal_volts : 12.0,
		ideal_amps : 30.0,
		amp_hours : 18,
		charge_percent : getprop("/systems/electrical/battery-charge-percent") or 1.0,
		charge_amps : 7.0 };
		setprop("/systems/electrical/battery-charge-percent", obj.charge_percent);
		return obj;
}

##
# Passing in positive amps means the battery will be discharged.
# Negative amps indicates a battery charge.
#

BatteryClass.apply_load = func (amps, dt) {
	var old_charge_percent = getprop("/systems/electrical/battery-charge-percent");
	
	if (getprop("/sim/freeze/replay-state"))
		return me.amp_hours * old_charge_percent;
	
	var amphrs_used = amps * dt / 3600.0;
	var percent_used = amphrs_used / me.amp_hours;
	
	var new_charge_percent = std.max(0.0, std.min(old_charge_percent - percent_used, 1.0));
	
	if (new_charge_percent < 0.1 and old_charge_percent >= 0.1)
		gui.popupTip("Warning: Low battery! Enable alternator or apply external power to recharge battery!", 10);
	me.charge_percent = new_charge_percent;
	setprop("/systems/electrical/battery-charge-percent", new_charge_percent);
	return me.amp_hours * new_charge_percent;
}

##
# Return output volts based on percent charged.  Currently based on a simple
# polynomial percent charge vs. volts function.
#

BatteryClass.get_output_volts = func {
	var x = 1.0 - me.charge_percent;
	var tmp = -(3.0 * x - 1.0);
	var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
	return me.ideal_volts * factor;
}


##
# Return output amps available.  This function is totally wrong and should be
# fixed at some point with a more sensible function based on charge percent.
# There is probably some physical limits to the number of instantaneous amps
# a battery can produce (cold cranking amps?)
#

BatteryClass.get_output_amps = func {
	var x = 1.0 - me.charge_percent;
	var tmp = -(3.0 * x - 1.0);
	var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
	return me.ideal_amps * factor;
}

##
# Set the current charge instantly to 100 %.
#

BatteryClass.reset_to_full_charge = func {
	me.apply_load(-(1.0 - me.charge_percent) * me.amp_hours, 3600);
}

##
# Alternator model class.
#

#	Engine Operation Manual 7.1.1:
#		voltage: 	12V
#		amperage:	20A	(maximal)

var AlternatorClass = {};

AlternatorClass.new = func {
	var obj = { parents : [AlternatorClass],
		rpm_source : "/engines/engine[0]/rpm",
		rpm_threshold : 800.0,
		ideal_volts : 14.0,
		ideal_amps : 12.0 };
		setprop( obj.rpm_source, 0.0 );
		return obj;
}

##
# Computes available amps and returns remaining amps after load is applied
#

AlternatorClass.apply_load = func( amps, dt ) {
	# Scale alternator output for rpms < 800.  For rpms >= 800
	# give full output.  This is just a WAG, and probably not how
	# it really works but I'm keeping things "simple" to start.
	var rpm = getprop( me.rpm_source );
	var factor = rpm / me.rpm_threshold;
	if ( factor > 1.0 ) {
		factor = 1.0;
	}
	# print( "alternator amps = ", me.ideal_amps * factor );
	var available_amps = me.ideal_amps * factor;
	return available_amps - amps;
}

##
# Return output volts based on rpm
#

AlternatorClass.get_output_volts = func {
	# scale alternator output for rpms < 800.  For rpms >= 800
	# give full output.  This is just a WAG, and probably not how
	# it really works but I'm keeping things "simple" to start.
	var rpm = getprop( me.rpm_source );
	var factor = rpm / me.rpm_threshold;
	if ( factor > 1.0 ) {
		factor = 1.0;
	}
	# print( "alternator volts = ", me.ideal_volts * factor );
	return me.ideal_volts * factor;
}


##
# Return output amps available based on rpm.
#

AlternatorClass.get_output_amps = func {
	# scale alternator output for rpms < 800.  For rpms >= 800
	# give full output.  This is just a WAG, and probably not how
	# it really works but I'm keeping things "simple" to start.
	var rpm = getprop( me.rpm_source );
	var factor = rpm / me.rpm_threshold;
	if ( factor > 1.0 ) {
		factor = 1.0;
	}
	# print( "alternator amps = ", ideal_amps * factor );
	return me.ideal_amps * factor;
}

var battery = BatteryClass.new();
var alternator = AlternatorClass.new();

var reset_battery_and_circuit_breakers = func {
	# Charge battery to 100 %
	battery.reset_to_full_charge();
	
	# Reset circuit breakers
	foreach( var b; keys(cb) ) {
		b.setBoolValue(1);
	}
}

##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
	new : func {
		var m = {
			parents: [ElectricalSystemUpdater]
		};
		# Request that the update function be called each frame
		m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
		return m;
	},
	
	enable: func {
		me.loop.reset();
		me.loop.enable();
	},
	
	disable: func {
		me.loop.disable();
	},
	
	reset: func {
		# Do nothing
	},
	
	update: func (dt) {
		update_virtual_bus(dt);
	}
};

##
# Model the system of relays and connections that join the battery,
# alternator, starter, master/alt switches, external power supply.
#

var update_virtual_bus = func (dt) {
	var serviceable = getprop("/systems/electrical/serviceable");
	var external_volts = 0.0;
	var load = 0.0;
	var battery_volts = 0.0;
	var alternator_volts = 0.0;
	if ( serviceable ) {
		battery_volts = battery.get_output_volts();
		alternator_volts = alternator.get_output_volts();
	}
	
	# switch state
	var master = master_switch.getBoolValue();
	if (getprop("/controls/electric/external-power"))
	{
		external_volts = 14;
	}
	
	# determine power source
	var bus_volts = 0.0;
	var power_source = nil;
	if ( master ) {
		bus_volts = battery_volts;
		power_source = "battery";
	}
	if ( master and (alternator_volts > bus_volts) ) {
		bus_volts = alternator_volts;
		power_source = "alternator";
	}
	if ( external_volts > bus_volts ) {
		bus_volts = external_volts;
		power_source = "external";
	}
	#print( "virtual bus volts = ", bus_volts );
	
	# bus network (1. these must be called in the right order, 2. the
	# bus routine itself determins where it draws power from.)
	load += soaring_bus( bus_volts );
	load += main_bus( bus_volts );
	
	# swtich the master breaker off if load is out of limits
	if ( load > 50 ) {
		cb.main.setBoolValue(0);
	}
	#print(load);
	# system loads and ammeter gauge
	var ammeter = 0.0;
	if ( bus_volts > 1.0 ) {
		# ammeter gauge
		if ( power_source == "battery" ) {
			ammeter = -load;
		} else {
			ammeter = battery.charge_amps;
		}
	}
	# print( "ammeter = ", ammeter );
	
	# charge/discharge the battery
	if ( power_source == "battery" ) {
		battery.apply_load( load, dt );
	} elsif ( bus_volts > battery_volts ) {
		battery.apply_load( -battery.charge_amps, dt );
	}
	
	# outputs
	setprop("/systems/electrical/amps", ammeter);
	setprop("/systems/electrical/volts", bus_volts);
	if (bus_volts > 12)
		vbus_volts = bus_volts;
	else
		vbus_volts = 0.0;
	
	return load;
}

#Load sources:
#	com:		https://www.skyfox.com/becker-ar6201-022-vhf-am-sprechfunkgeraet-8-33.html
#	vario:		http://www.ilec-gmbh.com/ilec/manuals/SC7pd.pdf
#	flarm:		http://flarm.com/wp-content/uploads/man/FLARM_InstallationManual_D.pdf
#	flarm display:	https://www.air-store.eu/Display-V3-FLARM

var soaring_bus = func( bv ) {
	var bus_volts = bv;
	var load = 0.0;
	# The soaring bus only powers essential equipment, namely COM1 and the electric vario (not installed)
	
	# Radio
	if( cb.comm0.getBoolValue() ){
		setprop("/systems/electrical/outputs/comm", bus_volts);
		if(com_ptt.getBoolValue() and com_start.getValue()==1){
			load += 19.2 / bus_volts;
		}else{
			load += 1.02*com_start.getValue() / bus_volts;
		}
	}else{
		setprop("/systems/electrical/outputs/comm", 0.0);
	}
	
}


var main_bus = func( bv ) {
	var bus_volts = 0.0;
	var load = 0.0;
	# The main bus powers everything else, it is activated when additionally the mode switch is placed in the "powered flight" position
	if ( mode_switch.getBoolValue() ){
		bus_volts = bv;
	}
	# Consumers:
	#	GPS					Garmin GPS100	http://retro-gps.info/Manuals/downloads/files/Garmin%20GPS%20100AVD_Inst.pdf / http://static.garmin.com/pumac/GPS100STD_OwnersManual.pdf
	#	XPDR					Garmin GTX328	http://static.garmin.com/pumac/GTX328Transponder_MaintenanceManual.pdf
	#	Electric Attitude Indicator		RCA 26EK	https://kellymfg.com/images/RCA26-EK_Information.pdf
	#	ACL (Anti-Collision Light/Strobe)	~3.5A
	#	Position Lights				~7.5A
	#	Landing Light (opt)			~7.5A
	#	starter					~75A
	#	fuel pumps (main 2.7A, boost +1.3A)
	#	TCU (0.3A)
	#	engine instruments (~2.0A)
	#		1. Fuel Quantity + Oil Temp
	#		2. Oil Pressure + CHT
	if ( cb.gps.getBoolValue() and bus_volts > 11 ) {
		setprop("/systems/electrical/outputs/gps", bus_volts);
		load += 5.5 / bus_volts; #500mA MAX at min voltage of 11V
	} else {
		setprop("/systems/electrical/outputs/gps", 0.0);
	}
	
	if ( cb.xpdr.getBoolValue() ) {
		setprop("/systems/electrical/outputs/transponder", bus_volts);
		if(xpdr_on.getBoolValue() and bus_volts != 0){
			load += 18 / bus_volts;	#1.5A at 12VDC
		}
	} else {
		setprop("/systems/electrical/outputs/transponder", 0.0);
	}
		
	if ( cb.att.getBoolValue() and bus_volts > 11) {
		setprop("/systems/electrical/outputs/attitude-indicator", bus_volts);
		setprop("/systems/electrical/outputs/attitude-indicator-suction-inhg", bus_volts/2);	#simulate suction, as this actually is an electrically-driven attitude gyro
		if( att_spin.getValue() <= 0.99 ) {
			load += 56 / bus_volts; #4.0A at 14VDC
		} else {
			load += 14 / bus_volts; #1.0A at 14VDC
		}
	} else {
		setprop("/systems/electrical/outputs/attitude-indicator", 0.0);
		setprop("/systems/electrical/outputs/attitude-indicator-suction-inhg", 0.0);
	}
	
	if ( acl_switch.getBoolValue() and bus_volts > 9 ) {
		setprop("/systems/electrical/outputs/anti-collision-light", bus_volts);
		load += 42 / bus_volts;
	} else {
		setprop("/systems/electrical/outputs/anti-collision-light", 0.0);
	}
	
	if ( pos_switch.getBoolValue() and bus_volts > 9 ) {
		setprop("/systems/electrical/outputs/navigation-lights", bus_volts);
		load += 90 / bus_volts;
	} else {
		setprop("/systems/electrical/outputs/navigation-lights", 0.0);
	}
	
	if ( land_switch.getBoolValue() and bus_volts > 9 ) {
		setprop("/systems/electrical/outputs/landing-light", bus_volts);
		load += 90 / bus_volts;
	} else {
		setprop("/systems/electrical/outputs/landing-light", 0.0);
	}
	
	if ( starter.getBoolValue() ) {
		setprop("systems/electrical/outputs/starter", bus_volts);
		if(bus_volts > 8){
			setprop("/controls/engines/engine[0]/starter", 1);
			load += 1050 / bus_volts; #~75A
		}else{
			setprop("/controls/engines/engine[0]/starter", 0);
		}
	} else {
		setprop("/systems/electrical/outputs/starter", 0.0);
		setprop("/controls/engines/engine[0]/starter", 0);
	}
	
	if ( bus_volts > 9 ){
		setprop("/systems/electrical/outputs/engine-fuel-pump", bus_volts);
		if ( fp_switch.getBoolValue() ) {
			setprop("/systems/electrical/outputs/boost-fuel-pump", bus_volts);
			load += 48 / bus_volts;
		} else {
			setprop("/systems/electrical/outputs/boost-fuel-pump", 0.0);
			load += 32.4;
		}
	} else {
		setprop("/systems/electrical/outputs/engine-fuel-pump", 0.0);
		setprop("/systems/electrical/outputs/boost-fuel-pump", 0.0);
	}
	
	if ( cb.engine1.getBoolValue() and bus_volts > 10.0 ) {
		setprop("/systems/electrical/outputs/fuel-quantity-indicator", bus_volts);
		setprop("/systems/electrical/outputs/oil-temp-indicator", bus_volts);
		load += 12 / bus_volts;
	} else {
		setprop("/systems/electrical/outputs/fuel-quantity-indicator", 0.0);
		setprop("/systems/electrical/outputs/oil-temp-indicator", 0.0);
	}
	
	if ( cb.engine2.getBoolValue() and bus_volts > 10.0 ) {
		setprop("/systems/electrical/outputs/oil-press-indicator", bus_volts);
		setprop("/systems/electrical/outputs/cht-indicator", bus_volts);
		load += 12 / bus_volts;
	} else {
		setprop("/systems/electrical/outputs/oil-press-indicator", 0.0);
		setprop("/systems/electrical/outputs/cht-indicator", 0.0);
	}
		
	# return cumulative load
	return load;
}


##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

# checking if battery should be automatically recharged
if (!getprop("/systems/electrical/save-battery-charge")) {
	battery.reset_to_full_charge();
};

system_updater.enable();

print("Electrical system initialized");

